#include "CameraCBuffer.hlsli"
#include "SamplerStates.hlsli"
#include "Common.hlsli"

Texture2D SourceTex : register(t0);
Texture1D<float> TangentialTex : register(t1);
Texture1D<float> DistortionTex : register(t2);

struct PSInput {
	float4 PositionViewportS : SV_Position;
	float2 TexCoords : TEXCOORD0;
};

// Num samples per blur direction (+ and - sagittal, + and - tangential)
// i.e. additional samples per pass = numSamples * 2;
static const int numSamples = 4;

float4 main(PSInput input) : SV_TARGET
{
	// Convert from texture space [0, 1] to film space [-FilmDimensions/2, FilmDimensions/2]
	float x = (input.TexCoords.x - 0.5) * FilmWidth;
	float y = (input.TexCoords.y - 0.5) * FilmHeight;
	float radius = sqrt(x * x + y * y);
	float coord1D = radius / (22 * kMmToMeters); // Convert from range of [0, 22] to [0, 1]

	// Get the distortion amount
	float distortionAmt = 0.01 * DistortionTex.SampleLevel(LinearSampler, coord1D, 0);
	float distortionFactor = 1 - distortionAmt;

	// Apply distortion
	float xNew = x * distortionFactor;
	float yNew = y * distortionFactor;
	radius *= distortionFactor;
	coord1D = radius / (22 * kMmToMeters);
	
	// Convert back to texture space to get sample point
	float u = xNew / FilmWidth + 0.5;
	float v = yNew / FilmHeight + 0.5;

	// Astigmatism - tangential blur
	float tangentialFactor = 0.01 * TangentialTex.SampleLevel(LinearSampler, coord1D, 0);

	// Sample the source image.
	float2 curTexCoords = float2(u, v);
	float4 color = SourceTex.SampleLevel(BorderedLinearSampler, curTexCoords, 0);
	int sampleCount = 1;

	// Sample the color buffer along the blur vector.
	if(tangentialFactor != 0) {
		float tangentialDeltaFactor = tangentialFactor / (numSamples * 2);
		// Tangential vector in texture space
		float2 tangentialVector = float2(-y / FilmHeight, x / FilmWidth);
		for(int i = 1; i <= numSamples; ++i)  
		{
			// Tangential samples
			float2 offset = i * tangentialDeltaFactor * tangentialVector;
			color += SourceTex.SampleLevel(BorderedLinearSampler, curTexCoords + offset, 0);
			color += SourceTex.SampleLevel(BorderedLinearSampler, curTexCoords - offset, 0);
		}
		sampleCount += numSamples * 2;
	}
	
	// Average all of the samples to get the final blur color.
	color /= sampleCount;

	return color;
}